/* =======================================================
   SCRIPT PRINCIPAL - Mon Site Web
   - Barre de navigation dynamique
   - Survol des images
   - Mode plein écran
   - Chargement des pages sans recharger (SPA légère)
   - Bouton Retour en Haut
======================================================= */

(function () {
  const navbar = document.getElementById('navbar');
  const main = document.querySelector('main');
  if (!navbar) return;

  // ---------- Variables scroll ----------
  let lastScroll = window.scrollY || 0;
  let ticking = false;
  const hideThreshold = 10;
  const minScrollToHide = 50;
  const paddingCorrection = 5;

  // ---------- Ajuste le padding-top de main ----------
  function updateMainOffset() {
    const navHeight = Math.ceil(navbar.getBoundingClientRect().height);
    main.style.paddingTop = (navHeight - paddingCorrection) + 'px';
  }

  // ---------- Gestion du scroll ----------
  function onScroll() {
    const currentScroll = window.scrollY || 0;
    const delta = currentScroll - lastScroll;

    if (currentScroll <= 0) {
      navbar.classList.add('visible');
      lastScroll = currentScroll;
      return;
    }

    if (Math.abs(delta) < hideThreshold) {
      lastScroll = currentScroll;
      return;
    }

    if (delta > 0 && currentScroll > minScrollToHide) {
      navbar.classList.remove('visible');
    } else if (delta < 0) {
      navbar.classList.add('visible');
    }

    lastScroll = currentScroll;
  }

  function requestTick() {
    if (!ticking) {
      window.requestAnimationFrame(() => {
        onScroll();
        ticking = false;
      });
      ticking = true;
    }
  }

  // ---------- Survol images ----------
  function setupHoverImages() {
    document.querySelectorAll('.change-on-hover').forEach(img => {
      const originalSrc = img.getAttribute('src');
      const hoverSrc = img.getAttribute('data-hover');
      if (!hoverSrc) return;

      img.addEventListener('mouseenter', () => img.setAttribute('src', hoverSrc));
      img.addEventListener('mouseleave', () => img.setAttribute('src', originalSrc));
      img.addEventListener('focus', () => img.setAttribute('src', hoverSrc));
      img.addEventListener('blur', () => img.setAttribute('src', originalSrc));
    });
  }

  // ---------- Mode plein écran ----------
  function setupFullscreen() {
    const fullscreenBtn = document.getElementById('fullscreen-btn');
    if (!fullscreenBtn) return;
    const img = fullscreenBtn.querySelector('img');
    const enterSrc = "images/pleinecrant01.png";
    const exitSrc = "images/pleinecrant02.png";

    fullscreenBtn.addEventListener('click', () => {
      if (!document.fullscreenElement) {
        document.documentElement.requestFullscreen()
          .then(() => img.src = exitSrc)
          .catch(console.error);
      } else {
        document.exitFullscreen()
          .then(() => img.src = enterSrc)
          .catch(console.error);
      }
    });
  }

  // ---------- Chargement des pages sans recharger ----------
  async function loadPage(url) {
    try {
      const response = await fetch(url);
      const html = await response.text();
      const content = document.getElementById('content');
      content.innerHTML = html;

      // Réassigner les événements sur les nouveaux liens
      content.querySelectorAll('[data-page]').forEach(link => {
        link.addEventListener('click', (e) => {
          e.preventDefault();
          loadPage(link.getAttribute('data-page'));
        });
      });

      // Recharger les images au survol pour la nouvelle page
      setupHoverImages();

      // Scroll en haut de la page après chargement
      window.scrollTo({ top: 0, behavior: 'smooth' });

    } catch (err) {
      document.getElementById('content').innerHTML =
        "<p style='color:red;'>Impossible de charger la page.</p>";
      console.error(err);
    }
  }

  // ---------- Bouton Retour en Haut ----------
  const backToTopImg = document.querySelector('.nav-right img[alt="haut de page"]');
  if (backToTopImg) {
    backToTopImg.style.cursor = 'pointer';
    backToTopImg.addEventListener('click', () => {
      window.scrollTo({ top: 0, behavior: 'smooth' });
    });
  }

  // ---------- Initialisation ----------
  window.addEventListener('load', () => {
    navbar.classList.add('visible');
    updateMainOffset();
    lastScroll = window.scrollY || 0;
    setupHoverImages();
    setupFullscreen();
  });

  window.addEventListener('resize', updateMainOffset);
  window.addEventListener('scroll', requestTick, { passive: true });

  // ---------- Boutons principaux du menu ----------
  document.querySelectorAll('[data-page]').forEach(link => {
    link.addEventListener('click', (e) => {
      e.preventDefault();
      loadPage(link.getAttribute('data-page'));
    });
  });

})();
